<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\LoanOffer;

class FixInterestRates extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'loans:fix-interest-rates';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fix incorrectly stored interest rates and recalculate total_amount_to_pay';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Fixing interest rates and total_amount_to_pay...');
        
        $loanOffers = LoanOffer::all();
        $fixedCount = 0;
        
        foreach ($loanOffers as $loanOffer) {
            $needsUpdate = false;
            
            // Check if interest rate is unreasonably high (likely stored as total interest amount instead of rate)
            if ($loanOffer->interest_rate > 100) {
                // Set to default 12% annual interest rate
                $loanOffer->interest_rate = 12;
                $needsUpdate = true;
                $this->line("Fixed interest rate for Loan #{$loanOffer->id} to 12%");
            }
            
            // Now recalculate total_amount_to_pay with correct interest rate
            $principal = $loanOffer->requested_amount ?? 0;
            $interestRate = $loanOffer->interest_rate ?? 12;
            $tenure = $loanOffer->tenure ?? 12;
            
            if ($principal > 0 && $tenure > 0) {
                // Monthly interest rate
                $monthlyRate = $interestRate / 100 / 12;
                
                // Calculate monthly payment using loan amortization formula
                // M = P * [r(1+r)^n] / [(1+r)^n - 1]
                if ($monthlyRate > 0) {
                    $monthlyPayment = ($principal * $monthlyRate * pow(1 + $monthlyRate, $tenure)) / 
                                     (pow(1 + $monthlyRate, $tenure) - 1);
                    
                    $totalAmountToPay = $monthlyPayment * $tenure;
                } else {
                    // If interest rate is 0, total = principal
                    $totalAmountToPay = $principal;
                }
                
                // Update if different
                if (abs($loanOffer->total_amount_to_pay - $totalAmountToPay) > 0.01) {
                    $loanOffer->total_amount_to_pay = $totalAmountToPay;
                    $needsUpdate = true;
                    $this->line("Updated total_amount_to_pay for Loan #{$loanOffer->id}: " . number_format($totalAmountToPay, 2));
                }
            }
            
            if ($needsUpdate) {
                $loanOffer->save();
                $fixedCount++;
            }
        }
        
        $this->info("Fixed {$fixedCount} loan offers.");
        
        // Display summary
        $this->line("\n=== Updated Summary ===");
        $stats = \DB::table('loan_offers')
            ->selectRaw('
                COUNT(*) as total_records,
                AVG(interest_rate) as avg_interest_rate,
                MIN(interest_rate) as min_interest_rate,
                MAX(interest_rate) as max_interest_rate,
                AVG(requested_amount) as avg_requested,
                AVG(total_amount_to_pay) as avg_total_to_pay,
                AVG(take_home_amount) as avg_take_home,
                AVG(tenure) as avg_tenure
            ')
            ->first();
        
        $this->table(
            ['Metric', 'Value'],
            [
                ['Total Records', number_format($stats->total_records)],
                ['Average Interest Rate', number_format($stats->avg_interest_rate, 2) . '%'],
                ['Min Interest Rate', number_format($stats->min_interest_rate, 2) . '%'],
                ['Max Interest Rate', number_format($stats->max_interest_rate, 2) . '%'],
                ['Average Tenure', number_format($stats->avg_tenure, 0) . ' months'],
                ['Average Requested Amount', 'TZS ' . number_format($stats->avg_requested, 2)],
                ['Average Total to Pay', 'TZS ' . number_format($stats->avg_total_to_pay, 2)],
                ['Average Take Home Amount', 'TZS ' . number_format($stats->avg_take_home, 2)],
            ]
        );
        
        return Command::SUCCESS;
    }
}